# Bogumił Kamiński, 2018-12-13
# Run under Julia 1.0.2

using DifferentialEquations # version: v5.3.1
using PyPlot # version: v2.6.3

function f(y, i, j, d)
    du = d * (1 - sum(y[l+1] for l in 0:(d-1)))
    s = sum(l*y[l+1] for l in 1:(d-1)) / du
    v = 0.0 - (i == j) + (i == 0)
    v += j * (d - 1) * ((i+1)*y[i+1+1] - i*y[i+1]) / du
    v += j * binomial(d-1, i) * s^(d-i-1) * (1-s)^i
    return v
end

function mat(y, d, k)
    m = fill(1.0, d-k, d-k)
    for i in 1:(d-k-1), j in 1:(d-k)
        m[i+1, j] = f(y, i, j, d)
    end
    m
end

function tau(y, d, k)
    m = mat(y, d, k)
    r = zeros(d-k)
    r[1] = 1.0
    m \ r
end

function dy(y, d, k)
    τ = tau(y, d, k)
    x = zeros(d+1)
    if k < d-1
        τ[d-k] ≤ 0 && return x
    else
        y[1] ≥ 1 && return x
    end
    for i in 0:(d-1), j in 1:(d-k)
        x[i+1] += τ[j] * f(y, i, j, d)
    end
    for i in 2:(d-k) # not really needed, but improves solution stability
        x[i] = 0.0
    end
    x
end

our_ode(du, u, p, t) = du .= dy(u, p[1], p[2])

function solprob(d)
    plt[:rc]("font", size=20)
    ycutoff = zeros(d+1)
    tcut = 0.0
    allsol = [copy(ycutoff)]
    allt = [0.0]
    for k in 1:(d-1)
        prob = ODEProblem(our_ode,ycutoff,(tcut,1.0),[d, k])
        sol = solve(prob, dtmax=0.00001)
        cutoff = if k < d-1
            findfirst(x->x<0, getindex.(tau.(sol.u, d, k), d-k)) - 1
        else
            findfirst(x->x>1, getindex.(sol.u, 1)) - 1
        end
        tcut = sol.t[cutoff]
        ycutoff = sol.u[cutoff]
        for i in 2:(d-k)
            ycutoff[i] = 0
        end
        allsol = [allsol; sol.u[2:cutoff]]
        allt = [allt; sol.t[2:cutoff]]
        plt[:axvline](x=tcut, color="lightgray", linestyle=":")
        plt[:text](tcut, 1-(k-1)/14, string(round(tcut, digits=5)))
    end

    for i in 1:d
        plot(allt, getindex.(allsol, i))
    end
end

for d in 3:14
    plt[:clf]()
    println("Solving for d=$d")
    @time solprob(d)
    savefig("d$d.png")
end
